#include <ansi_c.h>
#include <utility.h>
#include <gpib.h>
#include <formatio.h>
#include "kei197a.h"

/* Keithley 197a Autoranging Microvolt DMM ================================== */
/* 																			  */
/* Written on:    		 September, 25 1997                                   */
/*                                                                            */
/* Contact:              David Henning - Rotational Engineer                  */
/*                       Motorola Semiconductor Products Sector               */
/*						 Strategic Systems Technologies	- Tempe, AZ			  */
/*                    	 Pager	(602)498-4584								  */
/* ===========================================================================*/

/* ===========================================================================*/
/* NOTE TO USERS:  The Keithley 197a does not offer full programmability over */
/* the GPIB.  The measuring function cannot be programmed at all.  The measure*/
/* range cannot be programmed for current mode operation.  If you select a    */
/* new measuring function from the front panel (or perform any front panel    */
/* operation), it is best to close() and call the init() function again.             */
/* ===========================================================================*/

/* = INTERNAL FUNCTIONS ======================================================*/
int kei197a_check_status (int);
int kei197a_read_data (char *, int);
int kei197a_write_data (char *, int);
int kei197a_device_closed (void);
int kei197a_invalid_integer_range (int, int, int, int);
int kei197a_invalid_real_range (double, double, double, int);
int kei197a_convertFunctionCodeToString (int, char *);
int kei197a_convertRangeCodeToString (int, char *);
int kei197a_convertRelModeCodeToString (int, char *);  
int kei197a_convertTrigModeCodeToString (int, char *);  

/* = INTERNAL DATA ========================================================= */
static int bd;
static char cmd[6502];

/* the following four variables are used to monitor the state of the 197a */
static int kei197a_func; /* holds measuring function value from init() */
static int kei197a_trig_mode; /* holds command code that corresponds to trigger mode */
static int kei197a_range; /* holds command code that corresponds to range */
static int kei197a_rel_mode;

int kei197a_err;

/* ========================================================================= */
void kei197a_init (addr)
int addr;
{
int n;
int retval;

    if (kei197a_invalid_integer_range (addr, 0, 30,  -1) != 0)
        return;
    if (bd <= 0)  {
        bd = OpenDev ("", "kei197a");
        if (bd <= 0)  {
            kei197a_err = 220;
            return;
        }
    }
    if (ibpad (bd, addr) & 0x8000)  {
        kei197a_err = 233;
        return;
    }
    if (ibclr (bd) & 0x8000)  {
        kei197a_err = 236;
        return;
    }
    
    /* gets status word */
    if (kei197a_write_data ("U0X", 3) != 0) 
        return;
     
    if (kei197a_read_data (cmd, 40) != 0)
        return;
    
    /* status word
    Bytes  characters
    0-2		197
    3		(F)unction (0=DCV;1=ACV;2=Ohms;4=DCA;5=ACA;6=DC dB;7=DC db)
    4		(R)ange	 (See Manual for further explainations)
    5	    (Z)Relative
    6		(K)EOI
    7		(T)rigger
    8		(B)Data Buffer
    9		(Md)SRQ Data Mask
    10		(Me)SRQ Error Mask
    11		(Y)Terminator
    */
    
    /* Is this a 197? */
    retval = CompareBytes (cmd, 0, "197", 0, 3, 0);
	if (retval)	
		{
		kei197a_err = 223;
        n = CloseDev (bd);
        return;
        }
        
    /* What function is currently in use? */
    Fmt ((int*)&kei197a_func, "%i<%c",cmd[3] );
 	kei197a_func -= 48; // convert ascii to numeric value
 					
    /* Sets instrument to default state. */
    /* Measure function, dB Mode and range */
    /* are not set by this routine, however. */
    /* This will avoid any conflict with front-panel */
    /* switches that are selected. */
    
    n = kei197a_write_data ("G0X",3);
    n = kei197a_write_data ("Z0X",3);
    n = kei197a_write_data ("K0X",3);
    n = kei197a_write_data ("T0X",3);
    n = kei197a_write_data ("M0X",3);
    n = kei197a_write_data ("Y(CR LF)X",9);
    n = kei197a_write_data ("B0X",3);
    n = kei197a_read_data (cmd,40);
	
	/* G0 : Send prefix */      
	/* Z0 : Relative mode off */
	/* K0 : Send EOI */
	/* T0 : Trigger mode = Continuous on TALK */
	/* M0 : SRQ DISABLED */
	/* Y(CR LF): Terminator character = <CR><LF> */
	/* B0 : Data logger DISABLED */
}

/* ========================================================================= */
void kei197a_close (void)
{

    if (kei197a_device_closed () != 0)
        return;
        
    // place instrument in local mode
    if (ibloc (bd) & 0x8000)  {
        kei197a_err = 234;  
        return;
    }
    
    if (CloseDev (bd) < 0)
        kei197a_err = 221;	
    else
        kei197a_err = 0;
    bd = 0;
}

/* ========================================================================= */
void kei197a_queryMeasurementFunction (function_name, function_code)
char function_name[];
int * function_code;
/* Returns measurement function in easy-to-read string and DDC code form */
{
        
    if (kei197a_write_data ("U0X", 3) != 0) 
        return;
     
    /* Box might need a GET trigger! */
    /* It got the X above, it will get the TALK below */
    switch (kei197a_trig_mode)
    	{
    	case TRIG_ONESHOT_GET:
    	case TRIG_CONT_GET:
    		ibtrg(bd);
    		break;
    	}
    	
    if (kei197a_read_data (cmd, 40) != 0)
        return;
    
    /* What function is currently in use? */
    
    /* Measurement function resides in byte 3 of the status word */
    Fmt ((int*)&kei197a_func, "%i<%c",cmd[3] ); 
    
    /* convert ascii to numeric value corresponding to function code */
 	kei197a_func -= 48; 
 	
	kei197a_convertFunctionCodeToString (kei197a_func, function_name);
	
	*function_code = kei197a_func;
 	
}

/* ========================================================================= */
void kei197a_queryAllFunctions (funcstr, func, rangstr, rang, relstr, rel, trigstr, trig)
char funcstr[];
int * func;
char rangstr[];
int * rang;
char relstr[];
int * rel;
char trigstr[];
int * trig;
/* Returns state function, range, relative mode and trigger mode */
/* directly from instrument, not from static variables.  Updates */
/* static variables */

{   
    if (kei197a_write_data ("U0X", 3) != 0) 
        return;
    
    /* Box might need a GET trigger! */
    /* It got the X above, it will get the TALK below */
    switch (kei197a_trig_mode)
    	{
    	case TRIG_ONESHOT_GET:
    	case TRIG_CONT_GET:
    		ibtrg(bd);
    		break;
    	}
    	
    if (kei197a_read_data (cmd, 40) != 0)
        return;
    
    /*** DO FUNCTION ***/
    
    /* Measurement function resides in byte 3 of the status word */
    Fmt ((int*)&kei197a_func, "%i<%c",cmd[3] ); 
    
    /* convert ascii to numeric value corresponding to function code */
 	kei197a_func -= 48; 
 	
	kei197a_convertFunctionCodeToString (kei197a_func, funcstr);
	*func = kei197a_func;
	
	/*** DO RANGE ***/

    /* Range function resides in byte 4 of the status word */
    Fmt ((int*)&kei197a_range, "%i<%c",cmd[4] ); 
    
    /* convert ascii to numeric value corresponding to range code */
 	kei197a_range -= 48; 
 	
	kei197a_convertRangeCodeToString (kei197a_range, rangstr);
	*rang = kei197a_range;
	
	/*** DO RELATIVE MODE ***/

    /* Relative mode function resides in byte 5 of the status word */
    Fmt ((int*)&kei197a_rel_mode, "%i<%c",cmd[5] ); 
    
    /* convert ascii to numeric value corresponding to relmode code */
 	kei197a_rel_mode -= 48; 
 	
	kei197a_convertRelModeCodeToString (kei197a_rel_mode, relstr);
	*rel = kei197a_rel_mode;
	
	/*** DO TRIGGER MODE ***/

    /* Trigger mode function resides in byte 7 of the status word */
    Fmt ((int*)&kei197a_trig_mode, "%i<%c",cmd[7] ); 
    
    /* convert ascii to numeric value corresponding to function code */
 	kei197a_trig_mode -= 48; 
 	
	kei197a_convertTrigModeCodeToString (kei197a_trig_mode, trigstr);
	*trig = kei197a_trig_mode;
	
}	
/* ========================================================================= */
void kei197a_setdBMode (enable)
int enable;
/* This function enables or disables dB mode. */
/* The instrument must be in one of the following modes for */
/* command to be valid: DCV, ACV, DC dB or AC dB */

{
    int n;

    if (kei197a_device_closed () != 0)
        return;
  
    if ((kei197a_func < 2)||(kei197a_func > 5))
	    {
	    if (enable)
	    	n = kei197a_write_data ("D1X", 3);
	    else
	    	n = kei197a_write_data ("D0X", 3);
    	}
    else
    	kei197a_err = KEI197a_INVALID_dB_FUNCTION_REQUEST;
    return;
    
}
/* ========================================================================= */
void kei197a_setTriggerMode (trigger)
int trigger;
{
    int n;

    if (kei197a_device_closed () != 0)
        return;
    
    /* check that trigger value is valid */
    if (kei197a_invalid_integer_range (trigger, TRIG_CONT_TALK, TRIG_ONESHOT_X,  -3) != 0)
        return;
    kei197a_trig_mode = trigger;
    
    /* Build command */
	n = Fmt (cmd, "%s<T%dX", kei197a_trig_mode);	
	
	/* Send command over the bus */
    n = kei197a_write_data (cmd, NumFmtdBytes ());
    
    /* Must we do something here if T3X? */
}

/* =====================================================================*/
void kei197a_setMeasurementRange(range)
double range;
/* NOTE: Current ranges are not programmable on the 197a */
{
	int n;
	
	if (range < 0) range = 0 - range;  /* take abs(range) without math.h */
	
	switch (kei197a_func)  
    	{  
		case FUNC_DC_VOLTS:
			if (range == 0.0)
				kei197a_range = RANGE_AUTO;
			else if (range <= 0.2)
				kei197a_range = RANGE_200mV;
			else if (range <= 2.0)
				kei197a_range = RANGE_2V;
			else if (range <= 20.0)
				kei197a_range = RANGE_20V;
			else if (range <= 200.0)
				kei197a_range = RANGE_200V;
			else if (range <= 1000.0)
				kei197a_range = RANGE_1kV;
			else
				{
				kei197a_err = KEI197a_INVALID_RANGE;
				return;
				}
			break;
	
		case FUNC_AC_VOLTS:
			if (range == 0.0)
				kei197a_range = RANGE_AUTO;
			else if (range <= 0.2)
				kei197a_range = RANGE_200mV;
			else if (range <= 2.0)
				kei197a_range = RANGE_2V;
			else if (range <= 20.0)
				kei197a_range = RANGE_20V;
			else if (range <= 200.0)
				kei197a_range = RANGE_200V;
			else if (range <= 1000.0)
				kei197a_range = RANGE_1kV;
			else
				{
				kei197a_err = KEI197a_INVALID_RANGE;
				return;
				}
			break;
		case FUNC_DC_dB:
			if (range == 0.0)
				kei197a_range = RANGE_AUTO;
			else if (range <= 0.2)
				kei197a_range = RANGE_200mV;
			else if (range <= 2.0)
				kei197a_range = RANGE_2V;
			else if (range <= 20.0)
				kei197a_range = RANGE_20V;
			else if (range <= 200.0)
				kei197a_range = RANGE_200V;
			else if (range <= 1000.0)
				kei197a_range = RANGE_1kV;
			else
				{
				kei197a_err = KEI197a_INVALID_RANGE;
				return;
				}
			break;
			
		case FUNC_AC_dB:
			if (range == 0.0)
				kei197a_range = RANGE_AUTO;
			else if (range <= 0.2)
				kei197a_range = RANGE_200mV;
			else if (range <= 2.0)
				kei197a_range = RANGE_2V;
			else if (range <= 20.0)
				kei197a_range = RANGE_20V;
			else if (range <= 200.0)
				kei197a_range = RANGE_200V;
			else if (range <= 1000.0)
				kei197a_range = RANGE_1kV;
			else
				{
				kei197a_err = KEI197a_INVALID_RANGE;
				return;
				}
			break;
			
		case FUNC_OHMS:
			if (range == 0.0)
				kei197a_range = RANGE_AUTO;
		    else if (range <= 200.0)
				kei197a_range = RANGE_200_OHMS;
			else if (range <= 2000.0)
				kei197a_range = RANGE_2k_OHMS;
			else if (range <= 2.0E4)
				kei197a_range = RANGE_20k_OHMS;
			else if (range <= 2.0E5)
				kei197a_range = RANGE_200k_OHMS;
			else if (range <= 1.0E6)
				kei197a_range = RANGE_1M_OHMS;
			else
				{
				kei197a_err = KEI197a_INVALID_RANGE;
				return;
				}
			break;
			
		case FUNC_DC_AMPS:
			kei197a_err = KEI197a_AMPS_RANGE_NOT_PROGRAMMABLE;
			return;
			
		case FUNC_AC_AMPS:
			kei197a_err = KEI197a_AMPS_RANGE_NOT_PROGRAMMABLE;
			return;
			
		default:
			kei197a_err = KEI197a_INVALID_FUNCTION;
		}
	
	/* Build command */
	n = Fmt (cmd, "%s<R%dX", kei197a_range);	
	
	/* Send command over the bus */
    n = kei197a_write_data (cmd, 3);
    
}
/* ========================================================================= */
void kei197a_setRelativeMode (enable)
int enable;
/* This function enables or disables relative mode. */
/* The baseline suppression value is set based on a */
/* reading taken when the command is sent with enable=1. */
{
    int n;

    if (kei197a_device_closed () != 0)
        return;
    if (enable)
    	n = kei197a_write_data ("Z1X", 3);
    else
    	n = kei197a_write_data ("Z0X", 3);
    
}
								

/* ====================================================================== */
void kei197a_measure (reading)
double *reading;
{
    int n;

    if (kei197a_device_closed () != 0)
        return;
    switch (kei197a_trig_mode)  
    	{
        
        case TRIG_CONT_TALK:    /* Trigger box, TRIG on TALK doesn't work as far as I can tell */
        case TRIG_ONESHOT_TALK: /* As above ... */
        case TRIG_CONT_GET:		/* Send GET */
        case TRIG_ONESHOT_GET:
        	if (ibtrg (bd) & 0x8000)  
        		{
                kei197a_err = 235;
                return;
                }
            break;
        case TRIG_CONT_X:
        case TRIG_ONESHOT_X:
            if (kei197a_write_data ("X", 1) != 0)
                return;
            break;
        }
    
    /* address the instrument to talk */
    if (kei197a_read_data (cmd, 20) != 0)
        return;
    
    /* Parse floating point out of return value from instrument. */
    n = Scan (&cmd[4], "%s>%f", reading);
}
/* ========================================================================= */
void kei197a_sendTrigger (trig)
int trig;
/* Sends appropriate trigger to the instrument */
{
    int n;

    if (kei197a_device_closed () != 0)
        return;
    if (kei197a_invalid_integer_range (trig, 0, 2,  -2) != 0)
        return;
    
    switch (trig)  
    {
    case 1:  /* GET */
        if (ibtrg (bd) & 0x8000)  {
            kei197a_err = 235;
            return;
        	}
        break;
    case 2:  /* "X"-trigger */
        if (kei197a_write_data ("X", 1) != 0)
            return;
        break;
    }
}

/* ======================================================================= */
/************** HELPER FUNCTIONS GO DOWN HERE ******************************/
int kei197a_read_data (cmd, cnt)
char *cmd;
int cnt;
{
    int r__kei197a_read_data;

    if (ibrd (bd, cmd, (long)cnt) & 0x8000)
        kei197a_err = 231;
    else
        kei197a_err = 0;
    r__kei197a_read_data = kei197a_err;
    return r__kei197a_read_data;
}

/* ========================================================================= */
int kei197a_write_data (cmd, cnt)
char *cmd;
int cnt;
{
    int r__kei197a_write_data;

    if (ibwrt (bd, cmd, (long)cnt) & 0x8000)
        kei197a_err = 230;
    else
        kei197a_err = 0;
    r__kei197a_write_data = kei197a_err;
    return r__kei197a_write_data;
}

/* ========================================================================= */
int kei197a_device_closed (void)
{
    int r__kei197a_device_closed;

    if (bd <= 0)  {
        kei197a_err = 232;
        r__kei197a_device_closed = 1;
        return r__kei197a_device_closed;
    }
    r__kei197a_device_closed = 0;
    return r__kei197a_device_closed;
}

/* ========================================================================= */
int kei197a_invalid_integer_range (n, min, max, err_code)
int n;
int min;
int max;
int err_code;
{
    int r__kei197a_invalid_integer_rang;

    if (( -(n < min) |  -(n > max)) != 0)  {
        kei197a_err = err_code;
        r__kei197a_invalid_integer_rang = 1;
        return r__kei197a_invalid_integer_rang;
    }
    r__kei197a_invalid_integer_rang = 0;
    return r__kei197a_invalid_integer_rang;
}

/* ========================================================================= */
int kei197a_invalid_real_range (x, min, max, err_code)
double x;
double min;
double max;
int err_code;
{
    int r__kei197a_invalid_real_range;

    if (( -(x < min) |  -(x > max)) != 0)  {
        kei197a_err = err_code;
        r__kei197a_invalid_real_range = 1;
        return r__kei197a_invalid_real_range;
    }
    r__kei197a_invalid_real_range = 0;
    return r__kei197a_invalid_real_range;
}
/* ========================================================================== */
int kei197a_convertFunctionCodeToString(code, output)
int code;
char * output;

{
	switch (code)	
		{
		case FUNC_DC_VOLTS:
			strcpy (output, "DC Volts");
			break;
		case FUNC_AC_VOLTS:
			strcpy (output,"AC Volts");
			break;
		case FUNC_OHMS:
			strcpy (output, "OHMS");
			break;
		case FUNC_DC_AMPS:
			strcpy (output, "DC Amps");
			break;
		case FUNC_AC_AMPS:
			strcpy (output, "AC Amps");
			break;
		case FUNC_DC_dB:
			strcpy (output, "DC dB");
			break;
		case FUNC_AC_dB:
			strcpy (output, "AC dB");
			break;
		default:
			strcpy (output, "error");
			return (-1);
		}
	return (0);
}

int kei197a_convertRangeCodeToString(code, output)
int code;
char * output;

{	/*** VOLTS ***/
	if ((kei197a_func==FUNC_DC_VOLTS)||(kei197a_func==FUNC_AC_VOLTS))	
		switch(code)	
			{
			case RANGE_AUTO:
				strcpy (output, "Auto Range");
				break;
			case RANGE_200mV:
				strcpy (output,"200mV");
				break;
			case RANGE_2V:
				strcpy (output, "2V");
				break;
			case RANGE_20V:
				strcpy (output, "20V");
				break;
			case RANGE_200V:
				strcpy (output, "200V");
				break;
			case RANGE_1kV:
				strcpy (output, "1kV");
				break;
			default:
				strcpy (output, "error");
				return (-1);
			}
	/*** OHMS ***/
	else if ((kei197a_func==FUNC_OHMS))	
		switch(code)	
			{
			case RANGE_AUTO:
				strcpy (output, "Auto Range");
				break;
			case RANGE_200_OHMS:
				strcpy (output,"200 Ohms");
				break;
			case RANGE_2k_OHMS:
				strcpy (output, "2k Ohms");
				break;
			case RANGE_20k_OHMS:
				strcpy (output, "20k Ohms");
				break;
			case RANGE_200k_OHMS:
				strcpy (output, "200k Ohms");
				break;
			case RANGE_1M_OHMS:
				strcpy (output, "1M Ohm");
				break;
			default:
				strcpy (output, "error");
				return (-1);
			}
	/*** AMPS ***/		
	else if ((kei197a_func==FUNC_DC_AMPS)||(kei197a_func==FUNC_AC_AMPS))	
		switch(code)	
			{
			case RANGE_AUTO:
				strcpy (output, "Auto Range");
				break;
			case RANGE_200uA:
				strcpy (output,"200uA");
				break;
			case RANGE_2mA:
				strcpy (output, "2mA");
				break;
			case RANGE_20mA:
				strcpy (output, "20mA");
				break;
			case RANGE_200mA:
				strcpy (output, "200mA");
				break;
			case RANGE_2A:
				strcpy (output, "2A");
				break;
			case RANGE_10A:
				strcpy (output, "10A");
				break;
			default:
				strcpy (output, "error");
				return (-1);
			}
	
	return (0);
}

int kei197a_convertRelModeCodeToString(code, output)
int code;
char * output;

{
	switch (code)	
		{
		case 0: 
			strcpy (output, "off");
			break;
		case 1:
			strcpy (output,"ON");
			break;
		default:
			strcpy (output, "error");
			return (-1);
		}
	return (0);
}
	
int kei197a_convertTrigModeCodeToString (code, output)
int code;
char * output;

{
	switch (code)	
		{
		case TRIG_CONT_TALK: 
			strcpy (output, "Cont/Talk");
			break;
		case TRIG_ONESHOT_TALK:
			strcpy (output,"1-Shot/Talk");
			break;
		case TRIG_CONT_GET: 
			strcpy (output, "Cont/GET");
			break;
		case TRIG_ONESHOT_GET:
			strcpy (output,"1-Shot/GET");
			break;
		case TRIG_CONT_X: 
			strcpy (output, "Cont/X");
			break;
		case TRIG_ONESHOT_X:
			strcpy (output,"1-Shot/X");
			break;
		default:
			strcpy (output, "error");
			return (-1);
		}
	return (0);
}
